using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using Win32;

namespace DrvTestCS
{
    class Program
    {
        public enum MEDIA_TYPE
        {
            Unknown,                // Format is unknown 
            F5_1Pt2_512,            // 5.25", 1.2MB,  512 bytes/sector 
            F3_1Pt44_512,           // 3.5",  1.44MB, 512 bytes/sector 
            F3_2Pt88_512,           // 3.5",  2.88MB, 512 bytes/sector 
            F3_20Pt8_512,           // 3.5",  20.8MB, 512 bytes/sector 
            F3_720_512,             // 3.5",  720KB,  512 bytes/sector 
            F5_360_512,             // 5.25", 360KB,  512 bytes/sector 
            F5_320_512,             // 5.25", 320KB,  512 bytes/sector 
            F5_320_1024,            // 5.25", 320KB,  1024 bytes/sector 
            F5_180_512,             // 5.25", 180KB,  512 bytes/sector 
            F5_160_512,             // 5.25", 160KB,  512 bytes/sector 
            RemovableMedia,         // Removable media other than floppy 
            FixedMedia              // Fixed hard disk media 
        }

        // Data Structures 

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        public struct DISK_GEOMETRY
        {
            public long         Cylinders;
            public MEDIA_TYPE   MediaType;
            public uint         TracksPerCylinder;
            public uint         SectorsPerTrack;
            public uint         BytesPerSector; 
        }

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        public struct DISK_GEOMETRY_EX
        {
            public DISK_GEOMETRY Geometry;
            public long DiskSize;
            public byte Data;
        }

        /*** Partition information definition ***/
 
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        public struct PARTITION_INFORMATION 
        {
            public long StartingOffset;
            public long PartitionLength;
            public uint HiddenSectors;
            public uint PartitionNumber;
            public byte PartitionType;
            public bool BootIndicator;
            public bool RecognizedPartition;
            public bool RewritePartition;
        }

        /*** Partition information Ex definitions ***/

        public enum PARTITION_STYLE
        {
	        PARTITION_STYLE_MBR,
	        PARTITION_STYLE_GPT,
	        PARTITION_STYLE_RAW
        }	

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        public struct PARTITION_INFORMATION_MBR 
        {  
	        public byte PartitionType;
	        public bool BootIndicator;
	        public bool RecognizedPartition;
	        public uint HiddenSectors;
        }

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        unsafe public struct PARTITION_INFORMATION_GPT 
        {
	        public Guid  PartitionType;  
	        public Guid  PartitionId;  
	        public ulong Attributes;  
            public fixed char Name[36];
        } 

        [StructLayout(LayoutKind.Explicit, CharSet = CharSet.Auto)]
        public struct PARTITION_INFORMATION_EX
        {  
            // Be VERY careful when FieldOffsetAtrribute is used !
            [System.Runtime.InteropServices.FieldOffset(0)]
	        public PARTITION_STYLE PartitionStyle;

            [System.Runtime.InteropServices.FieldOffset(8)]
	        public long StartingOffset;

            [System.Runtime.InteropServices.FieldOffset(16)]
	        public long PartitionLength;

            [System.Runtime.InteropServices.FieldOffset(24)]
	        public uint PartitionNumber;  

            [System.Runtime.InteropServices.FieldOffset(28)]
	        public bool RewritePartition;  

            // union 
            [System.Runtime.InteropServices.FieldOffset(32)]
	        public PARTITION_INFORMATION_MBR Mbr;

            [System.Runtime.InteropServices.FieldOffset(32)]
            public PARTITION_INFORMATION_GPT Gpt;  
        }

        // Functions calling DeviceIoControl()
        
        static bool GetDriveGeometry(ref DISK_GEOMETRY dg, string sDriveName)
        {
	        int hDevice; // handle to the drive to be examined 
	        bool bOK; // results flag
	        int junk = 0; // discard results

	        hDevice = Win32IO.CreateFile(
		        sDriveName,	// drive to open
		        0,							// don't need any access to the drive
		        Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
		        (IntPtr) null,				// default security attributes
		        Win32IO.OPEN_EXISTING,		// disposition
		        0,							// file attributes
		        Win32IO.NULL);				// don't copy any file's attributes

		        if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
		        {
			        return false;
		        }
                byte[] pDataBuf = new byte[Marshal.SizeOf(dg)];
		        bOK = Win32IO.DeviceIoControl(
			        hDevice,						// device we are querying
			        Win32IO.IOCTL_DISK_GET_DRIVE_GEOMETRY(),// operation to perform
			        null, 0,						// no input buffer, so pass zero
                    pDataBuf, Marshal.SizeOf(dg),	// output buffer
			        ref junk,						// discard count of bytes returned
			        Win32IO.NULL);			        // synchronous I/O

                Win32IO.CloseHandle(hDevice); // we're done with the handle
                
                if (bOK)
                {
                    dg.Cylinders = Marshal.ReadInt64(pDataBuf, 0);
                    dg.MediaType = (MEDIA_TYPE) Marshal.ReadInt32(pDataBuf, 8);
                    dg.TracksPerCylinder =  (uint) Marshal.ReadInt32(pDataBuf, 12);
                    dg.SectorsPerTrack =    (uint)Marshal.ReadInt32(pDataBuf, 16);
                    dg.BytesPerSector =     (uint)Marshal.ReadInt32(pDataBuf, 20);
                }

		        return bOK;
        }

        static bool GetDriveGeometryEx(ref DISK_GEOMETRY_EX dgex, string sDriveName)
        {
	        int hDevice; // handle to the drive to be examined 
	        bool bOK; // results flag
	        int junk = 0; // discard results

	        hDevice = Win32IO.CreateFile(
		        sDriveName,	// drive to open
		        0,							// don't need any access to the drive
		        Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
		        (IntPtr) null,				// default security attributes
		        Win32IO.OPEN_EXISTING,		// disposition
		        0,							// file attributes
		        Win32IO.NULL);				// don't copy any file's attributes

		        if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
		        {
			        return false;
		        }

                byte[] pDataBuf = new byte[Marshal.SizeOf(dgex)];
		        bOK = Win32IO.DeviceIoControl(
			        hDevice,						// device we are querying
			        Win32IO.IOCTL_DISK_GET_DRIVE_GEOMETRY_EX(),	// operation to perform
			        null, 0,						// no input buffer, so pass zero
			        pDataBuf, Marshal.SizeOf(dgex),	// output buffer
			        ref junk,						// discard count of bytes returned
                    Win32IO.NULL);			        // synchronous I/O

                Win32IO.CloseHandle(hDevice); // we're done with the handle

                if (bOK)
                {
                    dgex.Geometry.Cylinders = Marshal.ReadInt64(pDataBuf, 0);
                    dgex.Geometry.MediaType = (MEDIA_TYPE)Marshal.ReadInt32(pDataBuf, 8);
                    dgex.Geometry.TracksPerCylinder = (uint)Marshal.ReadInt32(pDataBuf, 12);
                    dgex.Geometry.SectorsPerTrack = (uint)Marshal.ReadInt32(pDataBuf, 16);
                    dgex.Geometry.BytesPerSector = (uint)Marshal.ReadInt32(pDataBuf, 20);
                    dgex.DiskSize = Marshal.ReadInt64(pDataBuf, 24);
                    // Marshal the dgex.Data ?
                }
                return (bOK);
        }

        static bool GetPartitionInfo(ref PARTITION_INFORMATION pi, string sDiskName)
        {
	        int hDevice; // handle to the drive to be examined 
	        bool bOK;	// results flag
	        int junk = 0; // discard results

	        hDevice = Win32IO.CreateFile(
		        sDiskName,					// drive to open
                Win32IO.GENERIC_READ,				// don't need any access to the drive
                Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
                (IntPtr)null,						// default security attributes
                Win32IO.OPEN_EXISTING,				// disposition
		        0,							// file attributes
                Win32IO.NULL);						// don't copy any file's attributes

		        if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
		        {
			        return false;
		        }

                byte[] pDataBuf = new byte[Marshal.SizeOf(pi)];
                bOK = Win32IO.DeviceIoControl(
			        hDevice,						// device we are querying
                    Win32IO.IOCTL_DISK_GET_PARTITION_INFO(),	// operation to perform
			        null, 0,						// no input buffer, so pass zero
                    pDataBuf, Marshal.SizeOf(pi),			// output buffer
			        ref junk,							// discard count of bytes returned
                    Win32IO.NULL);			// synchronous I/O

                Win32IO.CloseHandle(hDevice); // we're done with the handle

                if (bOK)
                {
                    pi.StartingOffset = Marshal.ReadInt64(pDataBuf, 0);
                    pi.PartitionLength = Marshal.ReadInt64(pDataBuf, 8);
                    pi.HiddenSectors = (uint)Marshal.ReadInt32(pDataBuf, 16);
                    pi.PartitionNumber = (uint)Marshal.ReadInt32(pDataBuf, 20);
                    pi.PartitionType = Marshal.ReadByte(pDataBuf, 24);
                    pi.BootIndicator = Marshal.ReadByte(pDataBuf, 25) > 0 ? true : false;
                    pi.RecognizedPartition = Marshal.ReadByte(pDataBuf, 26) > 0 ? true : false;
                    pi.RewritePartition = Marshal.ReadByte(pDataBuf, 27) > 0 ? true : false;
                }
                return bOK;
        }

        unsafe static bool GetPartitionInfoEx(ref PARTITION_INFORMATION_EX piex, string sDiskName)
        {
	        int hDevice; // handle to the drive to be examined 
	        bool bOK;	// results flag
	        int junk = 0;		// discard results
            int i;

	        hDevice = Win32IO.CreateFile(
		        sDiskName,	// drive to open
                Win32IO.GENERIC_READ,				// don't need any access to the drive
                Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
                (IntPtr)null,						// default security attributes
                Win32IO.OPEN_EXISTING,				// disposition
		        0,							// file attributes
                Win32IO.NULL);						// don't copy any file's attributes

		        if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
		        {
			        return false;
		        }

                byte[] pDataBuf = new byte[144]; // 144 bytes is the size of Win32 structure
                bOK = Win32IO.DeviceIoControl(
			        hDevice,							// device we are querying
                    Win32IO.IOCTL_DISK_GET_PARTITION_INFO_EX(),	// operation to perform
			        null, 0,							// no input buffer, so pass zero
			        pDataBuf, 144,				// output buffer
			        ref junk,								// discard count of bytes returned
                    Win32IO.NULL);				// synchronous I/O

                Win32IO.CloseHandle(hDevice); // we're done with the handle

                if (bOK)
                {
                    piex.PartitionStyle = (PARTITION_STYLE) Marshal.ReadInt32(pDataBuf, 0);
                    piex.StartingOffset = Marshal.ReadInt64(pDataBuf, 8);
                    piex.PartitionLength = Marshal.ReadInt64(pDataBuf, 16);
                    piex.PartitionNumber = (uint)Marshal.ReadInt32(pDataBuf, 24);
                    piex.RewritePartition = Marshal.ReadByte(pDataBuf, 28) > 0 ? true : false;

                    switch (piex.PartitionStyle)
		            {
		            case PARTITION_STYLE.PARTITION_STYLE_MBR:
			            piex.Mbr.PartitionType = Marshal.ReadByte(pDataBuf, 32);
			            piex.Mbr.BootIndicator = Marshal.ReadByte(pDataBuf, 33) > 0 ? true : false;
			            piex.Mbr.RecognizedPartition = Marshal.ReadByte(pDataBuf, 34) > 0 ? true : false;
			            piex.Mbr.HiddenSectors = (uint) Marshal.ReadInt32(pDataBuf, 36);
                        break;

		            case PARTITION_STYLE.PARTITION_STYLE_GPT:
                        piex.Gpt.PartitionType = new Guid(
                            Marshal.ReadInt32(pDataBuf, 32),
                            Marshal.ReadInt16(pDataBuf, 36),
                            Marshal.ReadInt16(pDataBuf, 38),
                            Marshal.ReadByte (pDataBuf, 40),
                            Marshal.ReadByte (pDataBuf, 41),
                            Marshal.ReadByte (pDataBuf, 42),
                            Marshal.ReadByte (pDataBuf, 43),
                            Marshal.ReadByte (pDataBuf, 44),
                            Marshal.ReadByte (pDataBuf, 45),
                            Marshal.ReadByte (pDataBuf, 46),
                            Marshal.ReadByte (pDataBuf, 47));
                        piex.Gpt.PartitionId = new Guid(
                            Marshal.ReadInt32(pDataBuf, 48),
                            Marshal.ReadInt16(pDataBuf, 52),
                            Marshal.ReadInt16(pDataBuf, 54),
                            Marshal.ReadByte (pDataBuf, 56),
                            Marshal.ReadByte (pDataBuf, 57),
                            Marshal.ReadByte (pDataBuf, 58),
                            Marshal.ReadByte (pDataBuf, 59),
                            Marshal.ReadByte (pDataBuf, 60),
                            Marshal.ReadByte (pDataBuf, 61),
                            Marshal.ReadByte (pDataBuf, 62),
                            Marshal.ReadByte (pDataBuf, 63));
                        piex.Gpt.Attributes = (ulong) Marshal.ReadInt64(pDataBuf, 64);

                        for (i = 0; i < 36; i++)
                        {
                            fixed (PARTITION_INFORMATION_GPT* p = &piex.Gpt)
                            {
                                p->Name[i] = (char)Marshal.ReadInt16(pDataBuf, 72 + 2 * i);
                            }
                        }
                        break;
		            } 
                }
                return bOK;
        }

        static bool EjectMedia(string sDevice)
        {
            int hDevice; // handle to the drive to be examined 
            bool bOK; // results flag
            int junk = 0; // discard results

            hDevice = Win32IO.CreateFile(
                sDevice,	// drive to open
                Win32IO.GENERIC_READ,				// Generic read
                Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
                (IntPtr) null,						// default security attributes
                Win32IO.OPEN_EXISTING,				// disposition
                0,							// file attributes
                Win32IO.NULL);						// don't copy any file's attributes

            if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
            {
                return false;
            }

            bOK = Win32IO.DeviceIoControl(
                hDevice,						// device we are querying
                Win32IO.IOCTL_STORAGE_EJECT_MEDIA(),	// operation to perform
                (byte[]) null, 0,						// no input buffer, so pass zero
                null, 0,				// output buffer
                ref junk,							// discard count of bytes returned
                Win32IO.NULL);			// synchronous I/O

            Win32IO.CloseHandle(hDevice); // we're done with the handle

            return bOK;
        }

        static bool LoadMedia(string sDevice)
        {
            int hDevice; // handle to the drive to be examined 
            bool bOK; // results flag
            int junk = 0; // discard results

            hDevice = Win32IO.CreateFile(
                sDevice,	// drive to open
                Win32IO.GENERIC_READ,				// Generic read
                Win32IO.FILE_SHARE_READ | Win32IO.FILE_SHARE_WRITE, // share mode
                (IntPtr) null,						// default security attributes
                Win32IO.OPEN_EXISTING,				// disposition
                0,							        // file attributes
                Win32IO.NULL);						// don't copy any file's attributes

            if (hDevice == Win32IO.INVALID_HANDLE_VALUE) // we can't open the drive
            {
                return false;
            }

            bOK = Win32IO.DeviceIoControl(
                hDevice,						// device we are querying
                Win32IO.IOCTL_STORAGE_LOAD_MEDIA(),	// operation to perform
                (byte[]) null, 0,						// no input buffer, so pass zero
                null, 0,				// output buffer
                ref junk,							// discard count of bytes returned
                Win32IO.NULL);			// synchronous I/O

            Win32IO.CloseHandle(hDevice); // we're done with the handle

            return bOK;
        }

        static void PrintDriveGeometry(string sDriveName)
        {
            DISK_GEOMETRY dg = new DISK_GEOMETRY();
            DISK_GEOMETRY_EX dgex = new DISK_GEOMETRY_EX();
            long DiskSize;
            double dDiskSizeGB;

            bool bOK = GetDriveGeometryEx(ref dgex, sDriveName);

            if (bOK)
            {
                Console.WriteLine("*** GetDriveGeometryEx() results ***");
                Console.WriteLine("Cylinders = {0}", dgex.Geometry.Cylinders.ToString());
                Console.WriteLine("Media Type = {0}", dgex.Geometry.MediaType.ToString());
                Console.WriteLine("Tracks per cylinder = {0}", dgex.Geometry.TracksPerCylinder.ToString());
                Console.WriteLine("Sectors per track = {0}", dgex.Geometry.SectorsPerTrack.ToString());
                Console.WriteLine("Bytes per sector = {0}", dgex.Geometry.BytesPerSector.ToString());
                dDiskSizeGB = ((double)dgex.DiskSize) / (1024.0 * 1024.0 * 1024.0);
                Console.WriteLine("Disk size = {0} (Bytes) = {1} (MB) = {2:F} (GB)",
                    dgex.DiskSize, dgex.DiskSize / (1024 * 1024), dDiskSizeGB);
                Console.WriteLine("Note: Array Data was not analyzed! It contains further information.");
            }
            else
            {
                int Err = Marshal.GetLastWin32Error();
                Console.WriteLine("!!! GetDriveGeometryEx() failed. Error {0}.", Err);

                if (Err == Win32IO.ERROR_INVALID_FUNCTION)
                {
                    bOK = GetDriveGeometry(ref dg, "\\\\.\\PhysicalDrive0");
                    if (bOK)
                    {
                        Console.WriteLine("Cylinders = {0}", dg.Cylinders.ToString());
                        Console.WriteLine("Media Type = {0}", dg.MediaType.ToString());
                        Console.WriteLine("Tracks per cylinder = {0}", dg.TracksPerCylinder.ToString());
                        Console.WriteLine("Sectors per track = {0}", dg.SectorsPerTrack.ToString());
                        Console.WriteLine("Bytes per sector = {0}", dg.BytesPerSector.ToString());

                        DiskSize = dg.Cylinders * (long)dg.TracksPerCylinder *
                            (long)dg.SectorsPerTrack * (long)dg.BytesPerSector;
                        dDiskSizeGB = ((double)DiskSize) / (1024.0 * 1024.0 * 1024.0);
                        Console.WriteLine("Disk size = {0} (Bytes) = {1} (MB) = {2:F} (GB)\n",
                            DiskSize, DiskSize / (1024 * 1024), dDiskSizeGB);
                    }
                    else
                    {
                        Console.WriteLine("!!! GetDriveGeometry() failed. Error {0}.", Marshal.GetLastWin32Error());
                    }
                }
            }
        }

        static void PrintPartitionInfo(string sDiskName)
        {
            // partition information structure
            PARTITION_INFORMATION pi = new PARTITION_INFORMATION();
            // partition information ex structure
            PARTITION_INFORMATION_EX piex = new PARTITION_INFORMATION_EX();	
	        int dwErr = Win32IO.ERROR_SUCCESS;

	        bool bOK = GetPartitionInfoEx(ref piex, sDiskName);
	        if (bOK) 
	        {
		        Console.WriteLine("*** GetPartitionInfoEx() results ***");
		        Console.WriteLine("Partition Style     = {0}", piex.PartitionStyle.ToString());
		        Console.WriteLine("Starting Offset     = {0}", piex.StartingOffset.ToString());
		        Console.WriteLine("Partition Length    = {0}", piex.PartitionLength.ToString());
		        Console.WriteLine("Partition Number    = {0}", piex.PartitionNumber.ToString());
		        switch (piex.PartitionStyle)
		        {
		        case PARTITION_STYLE.PARTITION_STYLE_MBR:
			        Console.WriteLine("Partition Type      = {0}", piex.Mbr.PartitionType.ToString());
			        Console.WriteLine("Boot indicator      = {0}", piex.Mbr.BootIndicator.ToString());
			        Console.WriteLine("Recognized partition= {0}", piex.Mbr.RecognizedPartition.ToString());
			        Console.WriteLine("Hidden Sectors      = {0}", piex.Mbr.HiddenSectors.ToString());
			        break;

                case PARTITION_STYLE.PARTITION_STYLE_GPT:
			        Console.WriteLine("Partition Type      = {0}", piex.Gpt.PartitionType.ToString());
			        Console.WriteLine("Partition Id        = {0}", piex.Gpt.PartitionId.ToString());
			        Console.WriteLine("Attributes          = {0}", piex.Gpt.Attributes.ToString());
//!!!		        Console.WriteLine("Name                = '{0}'",  piex.Gpt.Name);
			        break;
		        } 
	        } 
	        else 
	        {
		        dwErr = Marshal.GetLastWin32Error();
		        Console.WriteLine("!!! GetPartitionInfoEx() failed. Error %ld.\n", dwErr);

		        if (dwErr == Win32IO.ERROR_INVALID_FUNCTION)
		        {
			        // MSDN says that IOCTL_DISK_GET_PARTITION_INFO is obsolete but better than FAIL!
			        bOK = GetPartitionInfo(ref pi, sDiskName);

			        if (bOK) 
			        {
				        Console.WriteLine("*** GetPartitionInfo() results ***\n");
				        Console.WriteLine("Starting Offset     = %I64d\n", pi.StartingOffset);
				        Console.WriteLine("Partition Length    = %I64d\n", pi.PartitionLength);
				        Console.WriteLine("Hidden Sectors      = %ld\n", pi.HiddenSectors);
				        Console.WriteLine("Partition Number    = %ld\n", pi.PartitionNumber);
				        Console.WriteLine("Partition Type      = %ld\n", pi.PartitionType);
				        Console.WriteLine("Boot indicator      = %ld\n", pi.BootIndicator);
				        Console.WriteLine("Recognized partition= %ld\n", pi.RecognizedPartition);
				        Console.WriteLine("Rewrite partition   = %ld\n", pi.RewritePartition);
			        } 
			        else 
			        {
				        Console.WriteLine("!!! GetPartitionInfo() failed. Error %ld.\n",
					        Marshal.GetLastWin32Error());
			        }
		        }
	        }
        }

        static void Main(string[] args)
        {
/*            
            bool bOK = EjectMedia("\\\\.\\E:");
            if (!bOK)
                Console.WriteLine("!!! EjectMedia() failed. Error {0}", Marshal.GetLastWin32Error());
            System.Threading.Thread.Sleep(2000);	// wait 2 seconds
            bOK = LoadMedia("\\\\.\\E:");
            if (!bOK)
                Console.WriteLine("!!! LoadMedia() failed. Error {0}", Marshal.GetLastWin32Error());
*/            
            PrintDriveGeometry("\\\\.\\PhysicalDrive0");
            PrintPartitionInfo("\\\\.\\C:");
        }
    }
}
