// DrvTestCPP.cpp : Several examples calling DeviceIoControl()
//
//	2007-03 Pavel Balda
//

#include "stdafx.h"
#include <windows.h>
#include <winioctl.h>
#include <tchar.h>

//#include "DrvTestCPP.h"

BOOL GetDriveGeometry(DISK_GEOMETRY *pdg, TCHAR* sDriveName)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK; // results flag
	DWORD junk; // discard results

	hDevice = CreateFile(
		sDriveName,	// drive to open
		0,							// don't need any access to the drive
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,						// device we are querying
			IOCTL_DISK_GET_DRIVE_GEOMETRY,	// operation to perform
			NULL, 0,						// no input buffer, so pass zero
			pdg, sizeof(*pdg),				// output buffer
			&junk,							// discard count of bytes returned
			(LPOVERLAPPED) NULL);			// synchronous I/O

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

BOOL GetDriveGeometryEx(DISK_GEOMETRY_EX *pdgex,  TCHAR* sDriveName)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK; // results flag
	DWORD junk; // discard results

	hDevice = CreateFile(
		sDriveName,	// drive to open
		0,							// don't need any access to the drive
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,						// device we are querying
			IOCTL_DISK_GET_DRIVE_GEOMETRY_EX,	// operation to perform
			NULL, 0,						// no input buffer, so pass zero
			pdgex, sizeof(*pdgex),				// output buffer
			&junk,							// discard count of bytes returned
			(LPOVERLAPPED) NULL);			// synchronous I/O

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

BOOL GetPartitionInfo(PARTITION_INFORMATION *ppi, TCHAR* sDiskName)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK;	// results flag
	DWORD junk;		// discard results

	hDevice = CreateFile(
		sDiskName,					// drive to open
		GENERIC_READ,				// don't need any access to the drive
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,						// device we are querying
			IOCTL_DISK_GET_PARTITION_INFO,	// operation to perform
			NULL, 0,						// no input buffer, so pass zero
			ppi, sizeof(*ppi),				// output buffer
			&junk,							// discard count of bytes returned
			(LPOVERLAPPED) NULL);			// synchronous I/O

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

BOOL GetPartitionInfoEx(PARTITION_INFORMATION_EX *ppiex, TCHAR* sDiskName)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK;	// results flag
	DWORD junk;		// discard results

	hDevice = CreateFile(
		sDiskName,	// drive to open
		GENERIC_READ,				// don't need any access to the drive
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,							// device we are querying
			IOCTL_DISK_GET_PARTITION_INFO_EX,	// operation to perform
			NULL, 0,							// no input buffer, so pass zero
			ppiex, sizeof(*ppiex),				// output buffer
			&junk,								// discard count of bytes returned
			(LPOVERLAPPED) NULL);				// synchronous I/O

		DWORD dwErr = GetLastError();

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

BOOL EjectMedia(TCHAR* sDevice)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK; // results flag
	DWORD junk; // discard results

	hDevice = CreateFile(
		sDevice,	// drive to open
		GENERIC_READ,				// Generic read
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,						// device we are querying
			IOCTL_STORAGE_EJECT_MEDIA,	// operation to perform
			NULL, 0,						// no input buffer, so pass zero
			NULL, 0,				// output buffer
			&junk,							// discard count of bytes returned
			(LPOVERLAPPED) NULL);			// synchronous I/O

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

BOOL LoadMedia(TCHAR* sDevice)
{
	HANDLE hDevice; // handle to the drive to be examined 
	BOOL bOK; // results flag
	DWORD junk; // discard results

	hDevice = CreateFile(
		sDevice,	// drive to open
		GENERIC_READ,				// Generic read
		FILE_SHARE_READ | FILE_SHARE_WRITE, // share mode
		NULL,						// default security attributes
		OPEN_EXISTING,				// disposition
		0,							// file attributes
		NULL);						// don't copy any file's attributes

		if (hDevice == INVALID_HANDLE_VALUE) // we can't open the drive
		{
			return (FALSE);
		}

		bOK = DeviceIoControl(
			hDevice,						// device we are querying
			IOCTL_STORAGE_LOAD_MEDIA,	// operation to perform
			NULL, 0,						// no input buffer, so pass zero
			NULL, 0,				// output buffer
			&junk,							// discard count of bytes returned
			(LPOVERLAPPED) NULL);			// synchronous I/O

		CloseHandle(hDevice); // we're done with the handle

		return bOK;
}

void printfGUID(TCHAR* sText, GUID* pGuid)
{
	TCHAR sGuid[128];
	sprintf(sGuid, _T("%s{%08X-%04X-%04X-%04X-%02X%02X%02X%02X%02X%02X}"), sText, 
		pGuid->Data1, pGuid->Data2, pGuid->Data3, (((WORD) pGuid->Data4[0]) << 8) | pGuid->Data4[1],
		pGuid->Data4[2], pGuid->Data4[3], pGuid->Data4[4], pGuid->Data4[5], pGuid->Data4[6],
		pGuid->Data4[7]);
	printf(_T("%s"), sGuid);
}

void PrintDriveGeometry(TCHAR* sDriveName)
{
	DISK_GEOMETRY dg;				// disk drive geometry structure
	DISK_GEOMETRY_EX dgex;			// disk drive geometry ex structure
	ULONGLONG DiskSize;				// size of the drive, in bytes
	double dDiskSizeGB;
	DWORD dwErr = ERROR_SUCCESS;

	BOOL bOK = GetDriveGeometryEx (&dgex, sDriveName);
	
	if (bOK)
	{
		printf("*** GetDriveGeometryEx() results ***\n");
		printf("Cylinders           = %I64d\n", dgex.Geometry.Cylinders);
		printf("Media Type          = %ld\n", dgex.Geometry.MediaType);
		printf("Tracks per cylinder = %ld\n", (ULONG) dgex.Geometry.TracksPerCylinder);
		printf("Sectors per track   = %ld\n", (ULONG) dgex.Geometry.SectorsPerTrack);
		printf("Bytes per sector    = %ld\n", (ULONG) dgex.Geometry.BytesPerSector);
		dDiskSizeGB = ((double) dgex.DiskSize.QuadPart)/(1024.0 * 1024.0 * 1024.0);
		printf("Disk size = %I64d (Bytes) = %I64d (MB) = %8.2f (GB)\n", dgex.DiskSize,
			dgex.DiskSize.QuadPart / (1024 * 1024), dDiskSizeGB);
		printf("Note: Array Data[1] was not analyzed! It contains further information.\n");
	}
	else
	{
		dwErr = GetLastError();
		printf ("!!! GetDriveGeometryEx() failed. Error %ld.\n", dwErr);

		if (dwErr == ERROR_INVALID_FUNCTION)
		{
			bOK = GetDriveGeometry (&dg, sDriveName);

			if (bOK) 
			{
				printf("*** GetDriveGeometry() results ***\n");
				printf("Cylinders           = %I64d\n", dg.Cylinders);
				printf("Media Type          = %ld\n", dg.MediaType);
				printf("Tracks per cylinder = %ld\n", (ULONG) dg.TracksPerCylinder);
				printf("Sectors per track   = %ld\n", (ULONG) dg.SectorsPerTrack);
				printf("Bytes per sector    = %ld\n", (ULONG) dg.BytesPerSector);

				DiskSize = dg.Cylinders.QuadPart * (ULONG)dg.TracksPerCylinder *
				(ULONG)dg.SectorsPerTrack * (ULONG)dg.BytesPerSector;
				dDiskSizeGB = ((double) (signed __int64) DiskSize)/(1024.0 * 1024.0 * 1024.0);
				printf("Disk size = %I64d (Bytes) = %I64d (MB) = %8.2f (GB)\n", DiskSize,
				DiskSize / (1024 * 1024), dDiskSizeGB);
			} 
			else 
			{
				printf ("!!! GetDriveGeometry() failed. Error %ld.\n", GetLastError ());
			}
		}
	}
}

void PrintPartitionInfoOffsets()
{
	PARTITION_INFORMATION pi;
	printf("PARTITION_INFORMATION Offsets and sizes\n");
	printf("StartingOffset         : %i  %i\n", (BYTE*) &pi.StartingOffset - (BYTE*) &pi, sizeof(pi.StartingOffset));
	printf("PartitionLength        : %i  %i\n", (BYTE*) &pi.PartitionLength - (BYTE*) &pi, sizeof(pi.PartitionLength));
	printf("HiddenSectors          : %i  %i\n", (BYTE*) &pi.HiddenSectors - (BYTE*) &pi, sizeof(pi.HiddenSectors));
	printf("PartitionNumber        : %i  %i\n", (BYTE*) &pi.PartitionNumber - (BYTE*) &pi, sizeof(pi.PartitionNumber));
	printf("PartitionType          : %i  %i\n", (BYTE*) &pi.PartitionType - (BYTE*) &pi, sizeof(pi.PartitionType));
	printf("BootIndicator          : %i  %i\n", (BYTE*) &pi.BootIndicator - (BYTE*) &pi, sizeof(pi.BootIndicator));
	printf("RecognizedPartition    : %i  %i\n", (BYTE*) &pi.RecognizedPartition - (BYTE*) &pi, sizeof(pi.RecognizedPartition));
	printf("RewritePartition       : %i  %i\n", (BYTE*) &pi.RewritePartition - (BYTE*) &pi, sizeof(pi.RewritePartition));
}
void PrintPartitionInfoExOffsets()
{
	PARTITION_INFORMATION_EX piex;
	printf("PARTITION_INFORMATION_EX Offsets and sizes\n");
	printf("PartitionStyle           : %i  %i\n", (BYTE*) &piex.PartitionStyle - (BYTE*) &piex, sizeof(piex.PartitionStyle));
	printf("StartingOffset           : %i  %i\n", (BYTE*) &piex.StartingOffset - (BYTE*) &piex, sizeof(piex.StartingOffset));
	printf("PartitionLength          : %i  %i\n", (BYTE*) &piex.PartitionLength - (BYTE*) &piex, sizeof(piex.PartitionLength));
	printf("PartitionNumber          : %i  %i\n", (BYTE*) &piex.PartitionNumber - (BYTE*) &piex, sizeof(piex.PartitionNumber));
	printf("RewritePartition         : %i  %i\n", (BYTE*) &piex.RewritePartition - (BYTE*) &piex, sizeof(piex.RewritePartition));
	printf("PARTITION_INFORMATION_MBR: %i  %i\n", (BYTE*) &piex.Mbr - (BYTE*) &piex, sizeof(piex.Mbr));
	printf("PARTITION_INFORMATION_GPT: %i  %i\n", (BYTE*) &piex.Gpt - (BYTE*) &piex, sizeof(piex.Gpt));
}
void PrintPartitionInfo(TCHAR* sDiskName)
{
	PARTITION_INFORMATION pi;		// partition information structure
	PARTITION_INFORMATION_EX piex;	// partition information ex structure
	DWORD dwErr = ERROR_SUCCESS;

	BOOL bOK = GetPartitionInfoEx(&piex, sDiskName);
	if (bOK) 
	{
		TCHAR* sPartitionStyles[] = {"MBR", "GPT", "RAW"};
		printf("*** GetPartitionInfoEx() results ***\n");
		printf("Partition Style     = %s\n", sPartitionStyles[piex.PartitionStyle]);
		printf("Starting Offset     = %I64d\n", piex.StartingOffset);
		printf("Partition Length    = %I64d\n", piex.PartitionLength);
		printf("Partition Number    = %ld\n", (ULONG) piex.PartitionNumber);
		switch (piex.PartitionStyle)
		{
		case PARTITION_STYLE_MBR:
			printf("Partition Type      = %ld\n", (ULONG) piex.Mbr.PartitionType);
			printf("Boot indicator      = %ld\n",  (ULONG) piex.Mbr.BootIndicator);
			printf("Recognized partition= %ld\n", (ULONG) piex.Mbr.RecognizedPartition);
			printf("Hidden Sectors      = %ld\n", (ULONG) piex.Mbr.HiddenSectors);
			break;

		case PARTITION_STYLE_GPT:
			printfGUID("Partition Type  = ", &piex.Gpt.PartitionType);
			printf("\n");
			printfGUID("Partition Id    = ", &piex.Gpt.PartitionId);
			printf("\n");
			printf("Attributes          = %uI64d\n", piex.Gpt.Attributes);
			printf("Name                = '%S'\n",  piex.Gpt.Name);
			break;
		} 

		printf("\n");
	} 
	else 
	{
		dwErr = GetLastError();
		printf ("!!! GetPartitionInfoEx() failed. Error %ld.\n", dwErr);

		if (dwErr == ERROR_INVALID_FUNCTION)
		{
			// MSDN says that IOCTL_DISK_GET_PARTITION_INFO is obsolete but better than FAIL!
			bOK = GetPartitionInfo(&pi, sDiskName);

			if (bOK) 
			{
				printf("*** GetPartitionInfo() results ***\n");
				printf("Starting Offset     = %I64d\n", pi.StartingOffset);
				printf("Partition Length    = %I64d\n", pi.PartitionLength);
				printf("Hidden Sectors      = %ld\n", (ULONG) pi.HiddenSectors);
				printf("Partition Number    = %ld\n", (ULONG) pi.PartitionNumber);
				printf("Partition Type      = %ld\n", (ULONG) pi.PartitionType);
				printf("Boot indicator      = %ld\n",  (ULONG) pi.BootIndicator);
				printf("Recognized partition= %ld\n", (ULONG) pi.RecognizedPartition);
				printf("Rewrite partition   = %ld\n", (ULONG)  pi.RewritePartition);
				printf("\n");
			} 
			else 
			{
				printf ("!!! GetPartitionInfo() failed. Error %ld.\n",
					GetLastError ());
			}
		}
	}
}

int main(int argc, char *argv[])
{
	BOOL bOK;					// generic results flag
	/*
	bOK = EjectMedia("\\\\.\\E:");
	if (!bOK)
		printf("!!! EjectMedia() failed. Error %ld.\n", GetLastError());
	Sleep(2000);	// wait 2 seconds
	bOK = LoadMedia("\\\\.\\E:");
	if (!bOK)
		printf("!!! LoadMedia() failed. Error %ld.\n", GetLastError());
	*/
	PrintDriveGeometry("\\\\.\\PhysicalDrive0");
	PrintPartitionInfo("\\\\.\\C:");
//	PrintPartitionInfoOffsets();
//	PrintPartitionInfoExOffsets();
	return 0;
}
